// ============================================================
// NAVTCC SEO Tool — Elementor JSON Template Engine
// Created by Jawad with help of AI
// ============================================================

const hexId      = () => Math.random().toString(16).slice(2, 10);
const encQ       = s  => encodeURIComponent(String(s || ''));

const unsplash    = (kw, w=1200, h=600) => `https://source.unsplash.com/${w}x${h}/?${encQ(kw)},professional`;
const unsplashSvc = (svc, type, w=900, h=400) => `https://source.unsplash.com/${w}x${h}/?${encQ(svc + ',' + type)}`;

const SVC_ICONS = ['fas fa-star','fas fa-check-circle','fas fa-cogs','fas fa-heart','fas fa-award','fas fa-shield-alt','fas fa-bolt','fas fa-gem','fas fa-trophy','fas fa-thumbs-up'];
const SVC_EMOJIS= ['⭐','✅','⚙️','❤️','🏆','🛡️','⚡','💎','🎯','🌟','🔥','✨'];

// ── Widget builders ──────────────────────────────────────────
const mkSection = (bgColor, elements, extra = {}) => ({
    id: hexId(), elType: 'section',
    settings: {
        background_background: 'classic',
        background_color: bgColor,
        padding: { unit:'px', top:'80', right:'20', bottom:'80', left:'20', isLinked:false },
        ...extra
    },
    elements: [{ id: hexId(), elType: 'column', settings: { _column_size: 100 }, elements }]
});

const mkH = (text, tag, color, size, align = 'center') => ({
    id: hexId(), elType: 'widget', widgetType: 'heading',
    settings: { title: text, header_size: tag, align, title_color: color, typography_font_size: { unit:'px', size } }
});

const mkTxt = (html, color = '#475569') => ({
    id: hexId(), elType: 'widget', widgetType: 'text-editor',
    settings: { editor: html, text_color: color }
});

const mkBtn = (text, url = '#contact', bg, textColor = '#ffffff', radius = '6px') => ({
    id: hexId(), elType: 'widget', widgetType: 'button',
    settings: {
        text, align: 'center', background_color: bg, button_text_color: textColor,
        border_radius: { unit:'px', top: radius.replace('px',''), right: radius.replace('px',''), bottom: radius.replace('px',''), left: radius.replace('px',''), isLinked: true },
        link: { url, is_external: '', nofollow: '' }
    }
});

const mkImg = (url, alt) => ({
    id: hexId(), elType: 'widget', widgetType: 'image',
    settings: { image: { url, alt }, image_size: 'full' }
});

const mkIcon = (icon, title, desc, color) => ({
    id: hexId(), elType: 'widget', widgetType: 'icon-box',
    settings: { icon: { value: icon, library: 'fa-solid' }, title_text: title, description_text: desc, icon_color: color }
});

const mkHtml = html => ({
    id: hexId(), elType: 'widget', widgetType: 'html', settings: { html }
});

// ── Schema builder ───────────────────────────────────────────
function buildSchema(biz) {
    return JSON.stringify({
        '@context': 'https://schema.org',
        '@type': biz.schemaType || 'LocalBusiness',
        name: biz.name,
        description: biz.metaDesc_home,
        telephone: biz.phone,
        email: biz.email,
        address: { '@type': 'PostalAddress', streetAddress: biz.address, addressLocality: biz.city, addressCountry: 'PK' },
        openingHours: biz.hours,
        url: `https://www.${slugify(biz.name)}.com`
    });
}

// ── HOME PAGE ────────────────────────────────────────────────
function buildHomePage(biz, theme, content) {
    const c      = content || {};
    const schema = buildSchema(biz);

    const sections = [
        // HERO
        mkSection(theme.heroGrad, [
            mkImg(unsplash(biz.type), `${biz.focusKeyword} — ${biz.name} ${biz.city}`),
            mkH(c.hero_headline || `${biz.focusKeyword} | ${biz.name}`, 'h1', '#ffffff', 46),
            mkH(c.hero_subheadline || biz.tagline, 'h2', 'rgba(255,255,255,0.88)', 20),
            mkBtn(c.hero_cta1 || 'Our Services', '#services', theme.accent, '#ffffff', theme.btnRadius),
            mkBtn(c.hero_cta2 || 'Contact Us',   '#contact',  'transparent', '#ffffff', theme.btnRadius),
        ]),

        // WHY US
        mkSection(theme.cardBg, [
            mkH(c.why_us_heading || `Why Choose ${biz.name} in ${biz.city}?`, 'h2', theme.primary, 34),
            ...(c.why_us_points || [
                { title: `Top Rated ${biz.type}`, desc: `Trusted by customers in ${biz.city}` },
                { title: 'Quality Guaranteed',    desc: 'Professional results every time' },
                { title: 'Always Available',       desc: biz.hours || 'Mon-Sat 9AM-6PM' }
            ]).map(p => mkIcon('fas fa-check-circle', p.title, p.desc, theme.accent)),
        ]),

        // SERVICES
        mkSection(theme.bg, [
            mkH(c.services_heading || `Our Services in ${biz.city}`, 'h2', theme.primary, 34),
            mkTxt(`<p style="text-align:center;color:#64748b">${c.services_intro || `${biz.name} provides professional ${biz.type.toLowerCase()} services in ${biz.city}.`}</p>`),
            ...(c.service_details || biz.services.map(s => ({ name: s, desc: `Professional ${s} services in ${biz.city}`, benefit: 'Quality Guaranteed' })))
                .map((s, i) => mkIcon(SVC_ICONS[i % SVC_ICONS.length], `${s.name} in ${biz.city}`, s.desc, theme.accent)),
        ]),

        // ABOUT
        mkSection(theme.cardBg, [
            mkH(c.about_heading || `Best ${biz.type} in ${biz.city}`, 'h2', theme.primary, 32, 'left'),
            mkTxt(`<div>${c.about_body || `<p>${biz.about || `${biz.name} is a leading ${biz.type.toLowerCase()} in ${biz.city}. ${biz.tagline}.`}</p>`}</div><ul style="margin-top:16px">
                ${(c.about_points || biz.keywords.map(k => `Expert ${k}`)).map(p => `<li style="padding:6px 0;color:#475569;list-style:none">✓ <strong>${esc(p)}</strong></li>`).join('')}
            </ul>`, theme.text),
            mkImg(unsplash(biz.type + ' team', 800, 500), `${biz.name} — ${biz.type} team in ${biz.city}`),
        ]),

        // TESTIMONIALS
        mkSection('#1e293b', [
            mkH(`Trusted by ${biz.city} Customers`, 'h2', '#ffffff', 28),
            mkHtml(`<div style="display:grid;grid-template-columns:1fr 1fr;gap:20px;max-width:720px;margin:0 auto">
                <blockquote style="background:rgba(255,255,255,0.06);border-left:3px solid ${theme.accent};border-radius:8px;padding:18px;color:rgba(255,255,255,0.82);font-style:italic;font-size:14px;line-height:1.7">
                    "${esc(c.testimonial_1 || `Best ${biz.type.toLowerCase()} in ${biz.city}! Highly professional team.`)}"
                    <br><strong style="color:${theme.accent};font-size:12px;font-style:normal;display:block;margin-top:10px">— Happy Customer, ${esc(biz.city)}</strong>
                </blockquote>
                <blockquote style="background:rgba(255,255,255,0.06);border-left:3px solid ${theme.accent};border-radius:8px;padding:18px;color:rgba(255,255,255,0.82);font-style:italic;font-size:14px;line-height:1.7">
                    "${esc(c.testimonial_2 || `Searched for ${biz.focusKeyword} and found ${biz.name} — the best decision.`)}"
                    <br><strong style="color:${theme.accent};font-size:12px;font-style:normal;display:block;margin-top:10px">— Satisfied Client, ${esc(biz.city)}</strong>
                </blockquote>
            </div>`),
        ]),

        // CTA
        mkSection(theme.heroGrad, [
            mkH(c.cta_heading || `Contact ${biz.name} Today`, 'h2', '#ffffff', 30),
            mkTxt(`<p style="text-align:center;color:rgba(255,255,255,0.85)">${esc(c.cta_body || biz.tagline)}</p>`),
            mkBtn(`📞 ${biz.phone}`, `tel:${biz.phone}`, theme.accent, '#ffffff', theme.btnRadius),
            mkBtn(`✉ Email Us`, `mailto:${biz.email}`, 'transparent', '#ffffff', theme.btnRadius),
        ]),

        // FOOTER
        mkSection(theme.primary, [
            mkH(biz.name, 'h3', '#ffffff', 20),
            mkTxt(`<p style="text-align:center;color:rgba(255,255,255,0.7)">📍 ${esc(biz.address)}<br>📞 ${esc(biz.phone)} &nbsp;|&nbsp; ✉ ${esc(biz.email)}</p>`),
            mkTxt(`<p style="text-align:center;color:rgba(255,255,255,0.5);font-size:13px">${esc(c.footer_tagline || biz.tagline)}</p>`),
            mkHtml(`<div style="text-align:center;margin-top:10px">
                <a href="/" style="color:rgba(255,255,255,0.6);margin:0 14px;text-decoration:none;font-size:14px">Home</a>
                <a href="/services" style="color:rgba(255,255,255,0.6);margin:0 14px;text-decoration:none;font-size:14px">Services</a>
                <a href="/contact" style="color:rgba(255,255,255,0.6);margin:0 14px;text-decoration:none;font-size:14px">Contact</a>
            </div>
            <p style="text-align:center;color:rgba(255,255,255,0.3);font-size:11px;margin-top:16px">© ${new Date().getFullYear()} ${esc(biz.name)}. All rights reserved. ${esc(biz.city)}, Pakistan</p>`),
            mkHtml(`<script type="application/ld+json">${schema}<\/script>`),
        ]),
    ];

    return JSON.stringify({
        version: '0.4', type: 'page',
        title: `${biz.name} — Home`,
        content: sections,
        page_settings: { post_title: biz.name, template: 'elementor_canvas' }
    }, null, 2);
}

// ── SERVICES PAGE ────────────────────────────────────────────
function buildServicesPage(biz, theme, content) {
    const c      = content || {};
    const schema = buildSchema(biz);
    const sections = [
        // HERO
        mkSection(theme.heroGrad, [
            mkHtml(`<p style="text-align:center;color:rgba(255,255,255,0.6);font-size:13px;margin-bottom:8px"><a href="/" style="color:${theme.accent}">Home</a> › Services</p>`),
            mkH(c.services_heading || `Our Services in ${biz.city} | ${biz.name}`, 'h1', '#ffffff', 42),
            mkTxt(`<p style="text-align:center;color:rgba(255,255,255,0.85)">${esc(c.services_page_intro || `${biz.name} provides expert ${biz.type.toLowerCase()} services in ${biz.city}.`)}</p>`),
        ]),
    ];

    // Individual service sections
    const svcDetails = c.service_details || biz.services.map(s => ({ name: s, desc: `Professional ${s} services in ${biz.city} by ${biz.name}. ${biz.tagline}.`, benefit: 'Quality Guaranteed' }));
    svcDetails.forEach((svc, i) => {
        sections.push(mkSection(i % 2 === 0 ? theme.bg : theme.cardBg, [
            mkH(`${svc.name} in ${biz.city}`, 'h2', theme.primary, 30, 'left'),
            mkImg(unsplashSvc(svc.name, biz.type), `${svc.name} in ${biz.city} — ${biz.name}`),
            mkTxt(`<p>${esc(svc.desc)}</p><p style="margin-top:12px;color:#475569">Contact ${esc(biz.name)} at ${esc(biz.phone)} to book your ${esc(svc.name.toLowerCase())} appointment in ${esc(biz.city)} today.</p>`),
            mkIcon('fas fa-check', `Quality ${svc.name}`, svc.benefit || 'Guaranteed results', theme.accent),
            mkIcon('fas fa-users', 'Expert Team', `Experienced ${biz.type.toLowerCase()} professionals`, theme.accent),
            mkIcon('fas fa-tag', 'Best Rates', `Affordable ${svc.name.toLowerCase()} in ${biz.city}`, theme.accent),
        ]));
    });

    sections.push(
        mkSection(theme.heroGrad, [
            mkH(c.cta_heading || `Book Your Appointment in ${biz.city}`, 'h2', '#ffffff', 28),
            mkBtn(`Contact ${biz.name}`, '/contact', theme.accent, '#ffffff', theme.btnRadius),
        ]),
        mkSection(theme.primary, [
            mkH(biz.name, 'h3', '#ffffff', 18),
            mkTxt(`<p style="text-align:center;color:rgba(255,255,255,0.65);font-size:13px">📞 ${esc(biz.phone)} &nbsp;|&nbsp; ✉ ${esc(biz.email)} &nbsp;|&nbsp; 📍 ${esc(biz.city)}</p>`),
            mkHtml(`<script type="application/ld+json">${schema}<\/script>`),
        ])
    );

    return JSON.stringify({
        version: '0.4', type: 'page',
        title: `Services — ${biz.name}`,
        content: sections,
        page_settings: { post_title: `Services | ${biz.name}`, template: 'elementor_canvas' }
    }, null, 2);
}

// ── CONTACT PAGE ─────────────────────────────────────────────
function buildContactPage(biz, theme, content) {
    const c      = content || {};
    const schema = buildSchema(biz);
    const mapQ   = encQ(`${biz.address} ${biz.city} Pakistan`);

    const sections = [
        mkSection(theme.heroGrad, [
            mkHtml(`<p style="text-align:center;color:rgba(255,255,255,0.6);font-size:13px;margin-bottom:8px"><a href="/" style="color:${theme.accent}">Home</a> › Contact</p>`),
            mkH(c.contact_heading || `Contact ${biz.name} in ${biz.city}`, 'h1', '#ffffff', 42),
            mkTxt(`<p style="text-align:center;color:rgba(255,255,255,0.85)">${esc(c.contact_intro || `We'd love to hear from you. Get in touch today.`)}</p>`),
        ]),
        mkSection(theme.bg, [
            mkIcon('fas fa-map-marker-alt', 'Our Address', biz.address, theme.accent),
            mkIcon('fas fa-phone',          'Call Us',     biz.phone,   theme.accent),
            mkIcon('fas fa-envelope',       'Email Us',    biz.email,   theme.accent),
            mkIcon('fas fa-clock',          'Working Hours', biz.hours || 'Mon-Sat 9AM-6PM', theme.accent),
        ]),
        mkSection(theme.cardBg, [
            mkHtml(`<iframe src="https://maps.google.com/maps?q=${mapQ}&output=embed" width="100%" height="400" frameborder="0" style="border:0;border-radius:12px;display:block" allowfullscreen loading="lazy"></iframe>`),
        ]),
        mkSection(theme.bg, [
            mkH('Send Us a Message', 'h2', theme.primary, 28),
            mkHtml(`<form style="max-width:540px;margin:0 auto;font-family:sans-serif" onsubmit="return false">
                <div style="margin-bottom:14px"><label style="display:block;font-weight:600;margin-bottom:6px;color:${theme.primary};font-size:13px">Full Name</label>
                <input type="text" placeholder="Your Name" style="width:100%;padding:11px;border:2px solid #e2e8f0;border-radius:7px;font-size:14px"></div>
                <div style="margin-bottom:14px"><label style="display:block;font-weight:600;margin-bottom:6px;color:${theme.primary};font-size:13px">Email</label>
                <input type="email" placeholder="your@email.com" style="width:100%;padding:11px;border:2px solid #e2e8f0;border-radius:7px;font-size:14px"></div>
                <div style="margin-bottom:14px"><label style="display:block;font-weight:600;margin-bottom:6px;color:${theme.primary};font-size:13px">Phone</label>
                <input type="tel" placeholder="+92-XXX-XXXXXXX" style="width:100%;padding:11px;border:2px solid #e2e8f0;border-radius:7px;font-size:14px"></div>
                <div style="margin-bottom:18px"><label style="display:block;font-weight:600;margin-bottom:6px;color:${theme.primary};font-size:13px">Message</label>
                <textarea placeholder="How can we help?" style="width:100%;padding:11px;border:2px solid #e2e8f0;border-radius:7px;font-size:14px;min-height:110px;resize:vertical"></textarea></div>
                <button type="submit" style="width:100%;padding:13px;background:${theme.accent};color:#fff;border:none;border-radius:${theme.btnRadius};font-weight:700;font-size:15px;cursor:pointer;${theme.btnExtra}">Send Message →</button>
                <p style="text-align:center;color:#94a3b8;font-size:12px;margin-top:10px">✓ We reply within 24 hours</p>
            </form>`),
            mkHtml(`<div style="background:${theme.bg};border:2px solid ${theme.accent}44;border-radius:10px;padding:18px;text-align:center;margin-top:24px;max-width:400px;margin-left:auto;margin-right:auto">
                <strong style="color:${theme.primary}">📍 Find us on Google Maps</strong><br>
                <span style="color:#64748b;font-size:13px">Search "<em>${esc(biz.name)} ${esc(biz.city)}</em>"</span>
            </div>`),
            mkHtml(`<script type="application/ld+json">${schema}<\/script>`),
        ]),
        mkSection(theme.primary, [
            mkH(biz.name, 'h3', '#ffffff', 18),
            mkTxt(`<p style="text-align:center;color:rgba(255,255,255,0.65);font-size:13px">📞 ${esc(biz.phone)} &nbsp;|&nbsp; ✉ ${esc(biz.email)} &nbsp;|&nbsp; 📍 ${esc(biz.address)}</p>`),
        ]),
    ];

    return JSON.stringify({
        version: '0.4', type: 'page',
        title: `Contact — ${biz.name}`,
        content: sections,
        page_settings: { post_title: `Contact | ${biz.name}`, template: 'elementor_canvas' }
    }, null, 2);
}

// ── Build all 3 pages ────────────────────────────────────────
function buildAllPages(biz, theme, content) {
    return {
        home:    buildHomePage(biz, theme, content),
        service: buildServicesPage(biz, theme, content),
        contact: buildContactPage(biz, theme, content),
    };
}
