// ============================================================
// NAVTCC SEO Tool — SEO Analyzer
// Created by Jawad with help of AI
// ============================================================

function calcSEO(biz) {
    const checks = [];
    const add = (label, pass, tip) => checks.push({ label, pass, tip });

    const titleWords = (biz.metaTitle_home || '').toLowerCase();
    const mainKw     = (biz.focusKeyword  || biz.keywords?.[0] || '').toLowerCase();
    const firstWord  = mainKw.split(' ')[0] || '';

    add(
        'H1 contains primary keyword',
        firstWord.length > 2 && titleWords.includes(firstWord),
        'Include your main keyword in the page H1 / meta title'
    );
    add(
        'Meta title 50–60 characters',
        (biz.metaTitle_home || '').length >= 45 && (biz.metaTitle_home || '').length <= 62,
        'Keep meta title between 50–60 characters for best Google display'
    );
    add(
        'Meta description 140–160 characters',
        (biz.metaDesc_home || '').length >= 130 && (biz.metaDesc_home || '').length <= 165,
        'Meta description should be 150–160 chars and include a call to action'
    );
    add(
        'City name in meta title',
        (biz.metaTitle_home || '').toLowerCase().includes((biz.city || '').toLowerCase()),
        'Add your target city to the meta title for local SEO'
    );
    add(
        'Phone number provided',
        (biz.phone || '').replace(/\D/g, '').length >= 10,
        'Phone number is required for NAP consistency and schema markup'
    );
    add(
        'Email address provided',
        (biz.email || '').includes('@') && (biz.email || '').includes('.'),
        'Email address helps with contact schema and trust signals'
    );
    add(
        'Physical address provided',
        (biz.address || '').trim().length > 10,
        'Full street address needed for Google Maps embed and LocalBusiness schema'
    );
    add(
        '3 or more keywords targeted',
        (biz.keywords || []).length >= 3,
        'Target at least 3 keywords for better search coverage'
    );
    add(
        '3 or more services listed',
        (biz.services || []).length >= 3,
        'More services = more keyword opportunities across pages'
    );
    add(
        'Tagline / value proposition set',
        (biz.tagline || '').trim().length > 5,
        'A clear tagline improves click-through rate from search results'
    );
    add(
        'Working hours provided',
        (biz.hours || '').trim().length > 3,
        'Working hours appear in LocalBusiness schema and GMB listing'
    );
    add(
        'Schema type identified',
        (biz.schemaType || '') !== '' && (biz.schemaType || '') !== 'LocalBusiness',
        'Specific schema type (e.g. Dentist, Restaurant) ranks better than generic LocalBusiness'
    );

    const passed = checks.filter(c => c.pass).length;
    const score  = Math.round((passed / checks.length) * 100);
    return { checks, score, passed, total: checks.length };
}

function calcDensity(biz, allPagesText) {
    const clean = allPagesText
        .toLowerCase()
        .replace(/<[^>]+>/g, ' ')
        .replace(/[^a-z0-9\s]/g, ' ')
        .replace(/\s+/g, ' ')
        .trim();

    const words = clean.split(' ').filter(Boolean);
    const wc    = words.length;

    return (biz.keywords || []).map(kw => {
        const kwClean = kw.toLowerCase().replace(/[^a-z0-9\s]/g, ' ').trim();
        const re      = new RegExp(kwClean.replace(/\s+/g, '\\s+'), 'gi');
        const count   = (clean.match(re) || []).length;
        const density = wc > 0 ? parseFloat(((count / wc) * 100).toFixed(2)) : 0;
        const status  = density < 0.4 ? 'low' : density > 3 ? 'high' : 'good';
        return { kw, count, density, status };
    });
}

// ── Render SEO Score tab ─────────────────────────────────────
function renderSEO(seoResult) {
    const sc      = seoResult.score;
    const scColor = sc >= 80 ? 'var(--green)' : sc >= 60 ? 'var(--amber)' : 'var(--red)';
    const scLabel = sc >= 80 ? 'Excellent!' : sc >= 60 ? 'Good — keep improving' : 'Needs Work';

    let html = `
        <div class="score-ring" style="border-color:${scColor}">
            <div class="score-num" style="color:${scColor}">${sc}</div>
            <div class="score-label">/ 100</div>
        </div>
        <div style="text-align:center;margin-bottom:24px;color:var(--muted);font-size:13px">
            ${seoResult.passed}/${seoResult.total} checks passed —
            <strong style="color:${scColor}">${scLabel}</strong>
        </div>`;

    seoResult.checks.forEach(c => {
        const col = c.pass ? 'rgba(16,185,129,0.2)' : 'rgba(244,63,94,0.1)';
        const ico = c.pass ? 'var(--green)' : 'var(--red)';
        html += `
        <div class="check-item" style="border-color:${col}">
            <div class="check-icon" style="background:${c.pass?'rgba(16,185,129,0.15)':'rgba(244,63,94,0.15)'};color:${ico}">
                ${c.pass ? '✓' : '✗'}
            </div>
            <div>
                <div class="check-title" style="color:${c.pass?'var(--text)':'#fca5a5'}">${esc(c.label)}</div>
                ${!c.pass ? `<div class="check-tip" style="color:var(--amber)">💡 ${esc(c.tip)}</div>` : ''}
            </div>
        </div>`;
    });

    document.getElementById('seoContent').innerHTML = html;
}

// ── Render Keywords tab ──────────────────────────────────────
function renderKeywords(kwData) {
    if (!kwData || kwData.length === 0) {
        document.getElementById('kwContent').innerHTML =
            '<div style="color:var(--muted);text-align:center;padding:40px;font-family:var(--font-mono)">No keywords found.</div>';
        return;
    }

    let html = '';
    kwData.forEach(k => {
        const barColor = k.status === 'good' ? 'var(--green)' : k.status === 'low' ? 'var(--amber)' : 'var(--red)';
        const pct      = Math.min(k.density * 20, 100);
        const note     = k.status === 'low'
            ? '⚠ Too low — add keyword to more headings and paragraphs'
            : k.status === 'good'
            ? '✓ Good density — keyword appears naturally'
            : '⚠ Too high — may look like keyword stuffing to Google';

        html += `
        <div class="kw-item">
            <div class="kw-top">
                <div class="kw-name">${esc(k.kw)}</div>
                <div style="display:flex;gap:10px;align-items:center">
                    <span style="font-size:12px;color:var(--muted);font-family:var(--font-mono)">${k.count}× found</span>
                    <span class="badge" style="background:${barColor}22;color:${barColor};border-color:${barColor}44">
                        ${k.density}% — ${k.status.toUpperCase()}
                    </span>
                </div>
            </div>
            <div class="kw-bar-bg">
                <div class="kw-bar" style="width:${pct}%;background:${barColor}"></div>
            </div>
            <div class="kw-note">${note}</div>
        </div>`;
    });

    document.getElementById('kwContent').innerHTML = html;
}
