// ============================================================
// NAVTCC SEO Tool — Smart LLM Router
// Created by Jawad with help of AI
// ============================================================

class LLMRouter {

    constructor() {
        this.state = this.loadState();
        this.cleanOldData();
    }

    // ── Get ordered list of providers to try ─────────────────
    getQueue() {
        const queue = [];

        for (const [id, cfg] of Object.entries(LLM_PROVIDERS)) {
            const key = localStorage.getItem(cfg.keyStorage);
            if (!key || !key.startsWith(cfg.keyPrefix)) continue; // no valid key

            const score = this.score(id, cfg);
            if (score < 0) continue; // exhausted or broken

            queue.push({ id, cfg, key, score });
        }

        // Sort: highest score first
        queue.sort((a, b) => b.score - a.score);
        return queue;
    }

    // ── Score a provider (higher = prefer) ───────────────────
    score(id, cfg) {
        const s    = this.state[id] || {};
        const now  = Date.now();

        // Rate limited? Skip
        if (s.rateLimitUntil && now < s.rateLimitUntil) return -1;

        // Too many errors recently? Lower score heavily
        const recentErrors = (s.errorCount || 0);
        if (recentErrors >= 3) return -1;

        // Daily quota check
        const used  = s.tokensToday || 0;
        const limit = cfg.tokensPerDay;
        const pct   = used / limit;
        if (pct >= 0.97) return -1; // quota exhausted

        // Calculate score
        let score = 100;
        score += (10 - cfg.priority) * 15; // lower priority number = higher score
        score += cfg.quality * 5;
        score += cfg.speed   * 3;
        score -= pct         * 40;         // penalize high usage
        score -= recentErrors * 8;

        return score;
    }

    // ── Main call — tries providers in order ──────────────────
    async call(systemPrompt, userPrompt, onStatusChange) {
        const queue = this.getQueue();

        if (queue.length === 0) {
            console.log('⚠ No API keys available — using JS fallback');
            return null;
        }

        for (const { id, cfg, key } of queue) {
            onStatusChange?.(`🤖 Trying ${cfg.name}...`);
            console.log(`🚀 Router: trying ${cfg.name} (${id})`);

            try {
                const result = await this.callProvider(cfg, key, systemPrompt, userPrompt);
                this.recordSuccess(id, result.tokensUsed || 3000);
                onStatusChange?.(`✅ ${cfg.name} responded`);
                console.log(`✅ Router: ${cfg.name} success`);
                return { data: result.data, provider: cfg.name, providerId: id };

            } catch (err) {
                console.warn(`⚠ Router: ${cfg.name} failed:`, err.message);

                if (err.isRateLimit) {
                    this.recordRateLimit(id);
                    onStatusChange?.(`⏳ ${cfg.name} rate limited, trying next...`);
                } else if (err.isAuthError) {
                    this.recordAuthError(id);
                    onStatusChange?.(`🔑 ${cfg.name} key invalid, trying next...`);
                } else {
                    this.recordError(id);
                    onStatusChange?.(`❌ ${cfg.name} error, trying next...`);
                }
                // Continue to next provider
            }
        }

        // All providers failed
        console.log('⚠ All providers failed — JS fallback');
        return null;
    }

    // ── Call a specific provider ──────────────────────────────
    async callProvider(cfg, key, systemPrompt, userPrompt) {
        if (cfg.format === 'gemini') {
            return await this.callGemini(cfg, key, systemPrompt, userPrompt);
        } else {
            return await this.callOpenAI(cfg, key, systemPrompt, userPrompt);
        }
    }

    async callOpenAI(cfg, key, systemPrompt, userPrompt) {
        const headers = {
            'Content-Type':  'application/json',
            'Authorization': 'Bearer ' + key,
            ...(cfg.extraHeaders || {})
        };

        const resp = await fetch(cfg.url, {
            method: 'POST',
            headers,
            body: JSON.stringify({
                model:           cfg.model,
                temperature:     0.82,
                max_tokens:      cfg.maxTokens,
                response_format: { type: 'json_object' },
                messages: [
                    { role: 'system', content: systemPrompt },
                    { role: 'user',   content: userPrompt   }
                ]
            })
        });

        const raw = await resp.json();

        if (resp.status === 401 || resp.status === 403) {
            const e = new Error(`Invalid API key for ${cfg.name}`);
            e.isAuthError = true; throw e;
        }
        if (resp.status === 429) {
            const e = new Error(`Rate limited: ${cfg.name}`);
            e.isRateLimit = true; throw e;
        }
        if (!resp.ok) {
            throw new Error(raw?.error?.message || `${cfg.name} HTTP ${resp.status}`);
        }

        const text       = raw.choices?.[0]?.message?.content;
        const tokensUsed = raw.usage?.total_tokens || 3000;
        if (!text) throw new Error(`${cfg.name} returned empty response`);

        return { data: this.parseJSON(text, cfg.name), tokensUsed };
    }

    async callGemini(cfg, key, systemPrompt, userPrompt) {
        const resp = await fetch(`${cfg.url}?key=${key}`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                contents: [{
                    role:  'user',
                    parts: [{ text: systemPrompt + '\n\n' + userPrompt }]
                }],
                generationConfig: {
                    temperature:      0.82,
                    maxOutputTokens:  cfg.maxTokens,
                    responseMimeType: 'application/json'
                }
            })
        });

        const raw = await resp.json();

        if (resp.status === 400 || resp.status === 403 || resp.status === 401) {
            const e = new Error(`Invalid Gemini API key`);
            e.isAuthError = true; throw e;
        }
        if (resp.status === 429) {
            const e = new Error(`Gemini rate limited`);
            e.isRateLimit = true; throw e;
        }
        if (!resp.ok) {
            throw new Error(raw?.error?.message || `Gemini HTTP ${resp.status}`);
        }

        const text       = raw.candidates?.[0]?.content?.parts?.[0]?.text;
        const tokensUsed = (raw.usageMetadata?.totalTokenCount) || 3000;
        if (!text) throw new Error('Gemini returned empty response');

        return { data: this.parseJSON(text, 'Gemini'), tokensUsed };
    }

    parseJSON(text, name) {
        if (typeof text !== 'string') return text;
        text = text.replace(/^```(?:json)?\s*/m, '').replace(/\s*```$/m, '').trim();
        try {
            return JSON.parse(text);
        } catch(e) {
            const m = text.match(/\{[\s\S]*\}/);
            if (m) {
                try { return JSON.parse(m[0]); } catch(e2) {}
            }
            throw new Error(`${name} returned invalid JSON. Please try again.`);
        }
    }

    // ── State management ──────────────────────────────────────
    recordSuccess(id, tokensUsed) {
        const s = this.state[id] || {};
        s.tokensToday  = (s.tokensToday  || 0) + tokensUsed;
        s.totalCalls   = (s.totalCalls   || 0) + 1;
        s.errorCount   = 0;
        s.lastSuccess  = Date.now();
        s.lastError    = null;
        if (!s.resetAt) s.resetAt = this.nextMidnight();
        this.state[id] = s;
        this.saveState();
    }

    recordRateLimit(id) {
        const s = this.state[id] || {};
        s.rateLimitUntil = Date.now() + 65000; // 65 sec cooldown
        s.errorCount     = (s.errorCount || 0) + 1;
        s.lastError      = Date.now();
        this.state[id]   = s;
        this.saveState();
    }

    recordAuthError(id) {
        const s = this.state[id] || {};
        s.errorCount   = (s.errorCount || 0) + 3; // penalize heavily
        s.lastError    = Date.now();
        this.state[id] = s;
        this.saveState();
    }

    recordError(id) {
        const s = this.state[id] || {};
        s.errorCount   = (s.errorCount || 0) + 1;
        s.lastError    = Date.now();
        this.state[id] = s;
        this.saveState();
    }

    // ── Get status for all providers (for UI) ─────────────────
    getStatus() {
        const result = {};
        const now    = Date.now();

        for (const [id, cfg] of Object.entries(LLM_PROVIDERS)) {
            const key   = localStorage.getItem(cfg.keyStorage);
            const s     = this.state[id] || {};
            const used  = s.tokensToday || 0;
            const limit = cfg.tokensPerDay;
            const pct   = Math.min(Math.round((used / limit) * 100), 100);

            let status = 'no-key';
            if (key && key.startsWith(cfg.keyPrefix)) {
                if (s.rateLimitUntil && now < s.rateLimitUntil) {
                    const secs = Math.ceil((s.rateLimitUntil - now) / 1000);
                    status = `rate-limit:${secs}`;
                } else if (pct >= 97) {
                    status = 'exhausted';
                } else if ((s.errorCount || 0) >= 3) {
                    status = 'errors';
                } else {
                    status = 'ready';
                }
            }

            result[id] = { status, pct, used, limit, name: cfg.name, emoji: cfg.emoji, label: cfg.label };
        }
        return result;
    }

    // Reset an individual provider (for testing)
    reset(id) {
        delete this.state[id];
        this.saveState();
    }
    resetAll() {
        this.state = {};
        this.saveState();
    }

    // ── Helpers ───────────────────────────────────────────────
    cleanOldData() {
        const now = Date.now();
        for (const id of Object.keys(this.state)) {
            const s = this.state[id];
            if (s.resetAt && now > s.resetAt) {
                // New day — reset daily counters
                s.tokensToday = 0;
                s.resetAt     = this.nextMidnight();
                // Also clear old rate limits
                if (s.rateLimitUntil && now > s.rateLimitUntil) {
                    delete s.rateLimitUntil;
                    s.errorCount = Math.max(0, (s.errorCount || 1) - 1);
                }
                this.state[id] = s;
            }
        }
        this.saveState();
    }

    nextMidnight() {
        const d = new Date();
        d.setHours(24, 0, 0, 0);
        return d.getTime();
    }

    loadState() {
        try {
            return JSON.parse(localStorage.getItem('navtcc_router_state') || '{}');
        } catch { return {}; }
    }
    saveState() {
        localStorage.setItem('navtcc_router_state', JSON.stringify(this.state));
    }
}

// Global singleton
const router = new LLMRouter();
