<?php
// ============================================================
// NAVTCC SEO Tool — API Proxy
// Created by Jawad with help of AI
// ============================================================

require_once 'config.php';

// ── CORS headers ─────────────────────────────────────────────
header('Content-Type: application/json');
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: SAMEORIGIN');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// ── Only POST allowed ────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    respond(405, 'Method not allowed');
}

// ── Parse JSON body ──────────────────────────────────────────
$body = file_get_contents('php://input');
$data = json_decode($body, true);
if (!$data) {
    respond(400, 'Invalid request body');
}

// ── Validate class code ──────────────────────────────────────
$classCode = trim($data['class_code'] ?? '');
if ($classCode !== CLASS_CODE) {
    respond(401, 'Invalid class code. Ask your instructor for the correct code.');
}

// ── Validate provider ────────────────────────────────────────
$provider = strtolower(trim($data['provider'] ?? ''));
if (!in_array($provider, ALLOWED_PROVIDERS)) {
    respond(400, 'Invalid provider. Choose gemini or groq.');
}

// ── Validate API key ─────────────────────────────────────────
$apiKey = trim($data['api_key'] ?? '');
if (empty($apiKey) || strlen($apiKey) > MAX_KEY_LENGTH) {
    respond(400, 'API key is missing or invalid.');
}
if ($provider === 'gemini' && !str_starts_with($apiKey, 'AIza')) {
    respond(400, 'Invalid Gemini key. Gemini keys start with: AIza');
}
if ($provider === 'groq' && !str_starts_with($apiKey, 'gsk_')) {
    respond(400, 'Invalid Groq key. Groq keys start with: gsk_');
}

// ── Validate prompt ──────────────────────────────────────────
$prompt = trim($data['prompt'] ?? '');
if (empty($prompt)) {
    respond(400, 'Prompt is empty.');
}
if (strlen($prompt) > MAX_PROMPT_LENGTH) {
    respond(400, 'Prompt too long. Max ' . MAX_PROMPT_LENGTH . ' characters.');
}
// Sanitize — strip any HTML/scripts from prompt
$prompt = strip_tags($prompt);

// ── Rate limiting by IP ──────────────────────────────────────
$ip       = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? 'unknown';
$ip       = explode(',', $ip)[0]; // take first IP if multiple
$ip       = preg_replace('/[^0-9a-fA-F:.]/', '', trim($ip));
$rateFile = sys_get_temp_dir() . '/navtcc_' . md5($ip) . '.json';
$now      = time();
$reqs     = [];

if (file_exists($rateFile)) {
    $saved = json_decode(file_get_contents($rateFile), true);
    if (is_array($saved)) {
        $reqs = array_values(array_filter($saved, fn($t) => ($now - $t) < RATE_LIMIT_WINDOW));
    }
}

if (count($reqs) >= RATE_LIMIT_REQUESTS) {
    $oldest   = min($reqs);
    $waitSecs = RATE_LIMIT_WINDOW - ($now - $oldest);
    $waitMins = ceil($waitSecs / 60);
    respond(429, "Rate limit reached. Please wait {$waitMins} minute(s). Max " . RATE_LIMIT_REQUESTS . " requests per hour.");
}

$reqs[] = $now;
file_put_contents($rateFile, json_encode($reqs), LOCK_EX);

// ── Log usage (counts only — no keys, no content) ────────────
logUsage($ip, $provider);

// ── Build system prompt ──────────────────────────────────────
$systemPrompt = buildSystemPrompt();

// ── Call API ─────────────────────────────────────────────────
if ($provider === 'gemini') {
    $result = callGemini($apiKey, $systemPrompt, $prompt);
} else {
    $result = callGroq($apiKey, $systemPrompt, $prompt);
}

if (isset($result['error'])) {
    respond(502, $result['error']);
}

// ── Return result ─────────────────────────────────────────────
http_response_code(200);
echo json_encode(['success' => true, 'data' => $result['data']]);
exit;

// ============================================================
// FUNCTIONS
// ============================================================

function callGemini(string $key, string $sys, string $prompt): array {
    $payload = json_encode([
        'contents' => [[
            'role'  => 'user',
            'parts' => [['text' => $sys . "\n\nBusiness details:\n\n" . $prompt]]
        ]],
        'generationConfig' => [
            'temperature'      => 0.85,
            'maxOutputTokens'  => 4096,
            'responseMimeType' => 'application/json'
        ]
    ]);

    $ch = curl_init(GEMINI_API_URL . '?key=' . urlencode($key));
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => $payload,
        CURLOPT_HTTPHEADER     => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT        => 60,
        CURLOPT_SSL_VERIFYPEER => true,
    ]);

    $raw    = curl_exec($ch);
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlErr= curl_error($ch);
    curl_close($ch);

    if ($raw === false || !empty($curlErr)) {
        return ['error' => 'Could not connect to Gemini API: ' . $curlErr];
    }

    $resp = json_decode($raw, true);

    if ($status === 429) {
        return ['error' => 'Gemini quota exceeded. Please try again later or switch to Groq.'];
    }
    if ($status === 403 || $status === 401) {
        return ['error' => 'Invalid Gemini API key. Please check your key at aistudio.google.com'];
    }
    if ($status !== 200) {
        $msg = $resp['error']['message'] ?? "Gemini API error (HTTP $status)";
        return ['error' => $msg];
    }

    $text = $resp['candidates'][0]['content']['parts'][0]['text'] ?? null;
    if (!$text) {
        return ['error' => 'Gemini returned empty response. Please try again.'];
    }

    return parseAIResponse($text);
}

function callGroq(string $key, string $sys, string $prompt): array {
    $payload = json_encode([
        'model'           => GROQ_MODEL,
        'temperature'     => 0.85,
        'max_tokens'      => 4096,
        'response_format' => ['type' => 'json_object'],
        'messages'        => [
            ['role' => 'system', 'content' => $sys],
            ['role' => 'user',   'content' => "Business details:\n\n" . $prompt]
        ]
    ]);

    $ch = curl_init(GROQ_API_URL);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => $payload,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $key
        ],
        CURLOPT_TIMEOUT        => 60,
        CURLOPT_SSL_VERIFYPEER => true,
    ]);

    $raw    = curl_exec($ch);
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlErr= curl_error($ch);
    curl_close($ch);

    if ($raw === false || !empty($curlErr)) {
        return ['error' => 'Could not connect to Groq API: ' . $curlErr];
    }

    $resp = json_decode($raw, true);

    if ($status === 429) {
        return ['error' => 'Groq rate limit reached. Please wait a moment and try again.'];
    }
    if ($status === 401 || $status === 403) {
        return ['error' => 'Invalid Groq API key. Please check your key at console.groq.com'];
    }
    if ($status !== 200) {
        $msg = $resp['error']['message'] ?? "Groq API error (HTTP $status)";
        return ['error' => $msg];
    }

    $text = $resp['choices'][0]['message']['content'] ?? null;
    if (!$text) {
        return ['error' => 'Groq returned empty response. Please try again.'];
    }

    return parseAIResponse($text);
}

function parseAIResponse(string $text): array {
    // Clean any accidental markdown fences
    $text = preg_replace('/^```(?:json)?\s*/m', '', $text);
    $text = preg_replace('/\s*```$/m', '', $text);
    $text = trim($text);

    $parsed = json_decode($text, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        // Try to extract JSON object
        preg_match('/\{[\s\S]*\}/m', $text, $matches);
        if (!empty($matches[0])) {
            $parsed = json_decode($matches[0], true);
        }
    }

    if (!$parsed || json_last_error() !== JSON_ERROR_NONE) {
        return ['error' => 'Could not parse AI response. Please try again.'];
    }

    return ['data' => $parsed];
}

function buildSystemPrompt(): string {
    return <<<'PROMPT'
You are an expert SEO copywriter and WordPress Elementor specialist for a digital marketing class in Pakistan.

Given business details, generate UNIQUE, NATURAL, SEO-optimized content for a 3-page website.

Return ONLY a raw JSON object — no markdown, no explanation, no code fences.

JSON structure required:
{
  "business": {
    "name": "string",
    "type": "string",
    "tagline": "string",
    "city": "string",
    "phone": "string",
    "email": "string",
    "address": "string",
    "hours": "string",
    "keywords": ["kw1","kw2","kw3","kw4"],
    "services": ["service1","service2","service3"],
    "schemaType": "e.g. Dentist / Restaurant / BeautySalon / LocalBusiness",
    "focusKeyword": "main keyword phrase",
    "metaTitle_home": "50-60 char title, keyword first",
    "metaDesc_home": "150-160 char meta with CTA",
    "metaTitle_services": "50-60 char services title",
    "metaDesc_services": "150-160 char services meta",
    "metaTitle_contact": "50-60 char contact title",
    "metaDesc_contact": "150-160 char contact meta",
    "slug_home": "/",
    "slug_services": "/services",
    "slug_contact": "/contact"
  },
  "content": {
    "hero_headline": "Creative H1 with primary keyword + city, max 10 words",
    "hero_subheadline": "One compelling tagline sentence",
    "hero_cta1": "Primary button text 3-4 words",
    "hero_cta2": "Secondary button text 3-4 words",
    "about_heading": "H2 with secondary keyword max 8 words",
    "about_body": "2 natural paragraphs 80-100 words total keyword-rich",
    "about_points": ["point 1 with keyword","point 2","point 3","point 4"],
    "why_us_heading": "H2 why choose us in city",
    "why_us_points": [
      {"title":"Feature 1","desc":"One sentence with keyword"},
      {"title":"Feature 2","desc":"One sentence"},
      {"title":"Feature 3","desc":"One sentence"}
    ],
    "services_heading": "H2 services in city with keyword",
    "services_intro": "One sentence intro for services",
    "service_details": [
      {"name":"Service Name","desc":"2-3 sentences natural keyword-rich description","benefit":"Key benefit phrase"}
    ],
    "testimonial_1": "Realistic customer review mentioning service and city 1-2 sentences",
    "testimonial_2": "Another realistic review different angle",
    "cta_heading": "Urgency H2 for CTA section",
    "cta_body": "1-2 sentences encouraging contact",
    "contact_heading": "H1 for contact page with city",
    "contact_intro": "1 welcoming sentence",
    "services_page_intro": "2-3 sentence keyword-rich intro for services page",
    "footer_tagline": "Short footer tagline 5-8 words"
  }
}

RULES:
- Keywords appear NATURALLY — never forced or stuffed
- Each service has unique helpful description
- Content feels human-written not robotic
- Pakistan context — city names, local tone
- H1 must contain primary keyword + city
- All meta titles 50-60 chars exactly
- All meta descriptions 150-160 chars exactly
- NAP consistent: same name/address/phone everywhere
- Schema type must match business precisely
PROMPT;
}

function logUsage(string $ip, string $provider): void {
    $logDir = dirname(LOG_FILE);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    $line = date('Y-m-d H:i:s') . ' | IP:' . md5($ip) . ' | Provider:' . $provider . PHP_EOL;
    file_put_contents(LOG_FILE, $line, FILE_APPEND | LOCK_EX);
}

function respond(int $code, string $message): never {
    http_response_code($code);
    echo json_encode(['error' => $message]);
    exit;
}
